<?php

namespace yidas\csv;

use Exception;

/**
 * CSV Writer
 * 
 * @author  Nick Tsai <myintaer@gmail.com>
 * @version 1.0.0
 */
class Writer
{
    // PHP csv handler uses UTF-8 encoding in default
    const DEFAULT_ENCODING = 'UTF-8';

    /**
     * Current PHP file pointer stream of the writer
     *
     * @var resource
     */
    protected $fileStream = null;

    /**
     * Controls when quotes should be always generated by the writer
     *
     * @var boolean
     */
    protected $quoteAll = true;

    /**
     * Controls which the encoding should be generated by the writer
     *
     * @var boolean
     */
    protected $encoding = null;

    /**
     * Constructor
     *
     * @param resource|string File pointer or filePath
     * @param array $optParams API Key or option parameters
     * @return self
     */
    function __construct($fileStream, $optParams=[]) 
    {
        // FilePath type input
        if (is_string($fileStream)) {
            // Check
            if (!file_exists($fileStream)) {
                throw new Exception("File not found at argument 1: {$fileStream}", 400);
            }

            $fileStream = fopen($fileStream, 'w');
        }
        // Check file stream
        if (!is_resource($fileStream)) {
            throw new Exception("Invalid file stream at argument 1", 400);
        }

        // Assignment
        $this->fileStream = $fileStream;
        $this->quoteAll = isset($optParams['quoteAll']) ? $optParams['quoteAll'] : $this->quoteAll;
        $this->encoding = isset($optParams['encoding']) ? $optParams['encoding'] : $this->encoding;

        return $this;
    }

    /**
     * Write the row parameter to the writer’s file stream, formatted according to the current setting.
     *
     * @param array $rowData
     * @return self 
     */
    public function writeRow($rowData)
    {
        
        if ($this->quoteAll==false && (!$this->encoding || $this->encoding==self::DEFAULT_ENCODING)) {
            
            fputcsv($this->fileStream, $rowData);

        } else {

            // Check rowData
            if (!is_array($rowData)) {
                throw new Exception("Invalid row data in writeRow()", 400);
            }

            foreach ($rowData as $key => $value) {
                $value = $this->_parseValue($value);
                $value = ($key==key($rowData)) ? $value : ",{$value}";
                fwrite($this->fileStream, $value);
            }
            fwrite($this->fileStream, "\n");
        }

        return $this;
    }

    /**
     * Write the rows parameter to the writer’s file stream, formatted according to the current setting.
     *
     * @param array $rowsData
     * @return self 
     */
    public function writeRows($rowsData)
    {
        
        foreach ($rowsData as $key => $rowData) {
            $this->writeRow($rowData);
        }
        
        return $this;
    }

    /**
     * Escape CSV value
     *
     * @param string $value
     * @return string
     */
    public static function escape($value)
    {
        return str_replace('"', '""', $value);
    }

    /**
     * Parse CSV value
     *
     * @param string $value
     * @return string
     */
    protected function _parseValue($value)
    {
        // Encoding
        if ($this->encoding && $this->encoding!=self::DEFAULT_ENCODING) {
            
            $value = mb_convert_encoding($value , $this->encoding, self::DEFAULT_ENCODING);
        }

        // Quoting
        if ($this->quoteAll || preg_match("(\"|,|\n|\r|\t| )", $value)===1) {
            $value = '"' . self::escape($value) . '"';
        }
        
        return $value;
    }

    /**
     * fclose for current file stream
     *
     * @return boolean Result
     */
    public function fclose()
    {
        return fclose($this->fileStream);
    }
}
